
///////////////////////////////////////////////////////////////////////////////
//
//   ROCKWELL SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   INITFUNC.C - Receiver Initialization Functions
// 
//
//   DESCRIPTION
//
//   Various functions used during receiver initialization (e.g., position and
//   time init, BIT, reset, cold start control, etc.).
//
//
//   REVISION HISTORY
//
//   $Log:   V:\Projects\Labmon\Source\Archives\INITFUNC.C_v  $
//   
//      Rev 1.17   07 Jan 2000 14:52:20   phungh
//   Labmon60 : Fix Y2K problem for default
//   initializing time prompt
//   
//      Rev 1.16   08 Dec 1999 15:58:48   phungh
//   Modify to allow options for invalidating
//   the eeprom in the reset command
//   
//      Rev 1.15   09 Nov 1998 10:47:40   phungh
//   labmon51: Fix file logging and missing
//   logged data bugs.  Changes made to the 
//   "Pause" variable ( 1 -> 0 ) to allow main
//   loop continue.  Move the "write to file"
//   code portion out of the interrupt handler
//   so that data is not missed because of time
//   spent too long in the interrupt handler
//   
//      Rev 1.4   Jul 09 1997 09:46:10   COLEJ
//   Multiple Changes...
//   
//      Rev 1.3   Feb 25 1997 14:26:52   COLEJ
//   Added 1208 timemark init stuff
//   
//      Rev 1.2   Feb 12 1997 16:05:02   COLEJ
//    
//   
//      Rev 1.1   Aug 13 1996 13:18:10   COLEJ
//    
//   
//      Rev 1.0   13 May 1996 14:52:32   GPSADMIN
//   Initial release to version control.
//
//
////////////////////////////////////////////////////////////////////////////////

#include <ctype.h>
#include <graph.h>
#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "LABMON.H"
#include "MESS_IO.H"
#include "MONMISC.H"
#include "DISPLAY.H"
#include "SENDNMEA.H"
#include "GENCONS.H"
#include "GENTYPES.H"
#include "ZCONS.H"
#include "ZTYPES.H"

void      make_tx_header( short Message, short WordCount );

extern tDATATYPE      DataType;

extern tMSGBUF        ZMsgBuf;
extern unsigned char  msgbuff[MAX_MSG_BUFFER];
extern unsigned char  rx_buff[MAX_RX_BUFFER+1];   
extern unsigned char  tx_buff[MAX_TX_BUFFER];
extern unsigned short head;

extern short          In_Nav_Mode; 
extern short          Auto_2D;
extern short          cmdline;   

extern double         lat, lon, alt;
extern double         ref_lat, ref_lon, ref_alt;

extern char           buff[80];
extern char           nmea_msg[80];

extern unsigned long cntgll, cntinit, cntibit; 
extern unsigned long cnt201, cnt202, cnt204, cnt206, cnt207, cnt208; 
extern unsigned long Cnt1200, Cnt1208, Cnt1212, Cnt1216, Cnt1219, Cnt1270;
extern unsigned long Cnt1300, Cnt1303, CntZQry;   

extern unsigned short   ColdStartDisable;
extern unsigned short   ColdStartTimeOut; 
extern unsigned short   StaticNavEnable;   

unsigned short ForceUse;
unsigned short MSLSelect;
unsigned short StoreRAM;
unsigned short StoreEEPROM;
unsigned short ClearRAM;
unsigned short ClearEEPROM;
double         AltStdDev;

short hours     = 0;
short minutes   = 0;
short day       = 0;
short month     = 0;
short year      = 0;
short seconds   = 0;
short utcoffset = 7;

short mask_num = 2;
float mask_angle;

extern unsigned short  SVs01To16;
extern unsigned short  SVs17To32;

double AltInput;


void alm_age_req( void )
{
   make_tx_header( 207, 5); 
   cnt207++;
   send_msg(tx_buff);
}

void alm_request( void )
{       
   switch(DataType){
      case NC_BINARY:  // navcore binary
         make_tx_header(206, 7);   
         *((short *)(tx_buff+10)) = 0;        /* Request Data For All SV's */
         cnt206++;
         send_msg(tx_buff);
         break; 
      case Z_BINARY:  // zodiac binary
         memset(&ZMsg.ZQuery, 0, sizeof(ZMsg.ZQuery));
         ZMsg.ZQuery.MsgID    = 1040;
         ZMsg.ZQuery.MsgFlags = Q_BIT | ANREQ;
         BldZQuery(&ZMsgBuf, &ZMsg.ZQuery);  
         memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.ZQuery) + HDR_LEN);
         CntZQry++;   
         send_msg(tx_buff); 
         break; 
   }
}

void eph_request( void )
{       
   switch(DataType){
      case NC_BINARY:  // navcore binary
         make_tx_header(206, 7);   
         *((short *)(tx_buff+10)) = 0;        /* Request Data For All SV's */
         cnt206++;
         send_msg(tx_buff);
         break; 
      case Z_BINARY:  // zodiac binary
           memset(&ZMsg.ZQuery, 0, sizeof(ZMsg.ZQuery));
           ZMsg.ZQuery.MsgID    = 1041;
           ZMsg.ZQuery.MsgFlags = Q_BIT | ANREQ;
           BldZQuery(&ZMsgBuf, &ZMsg.ZQuery);  
           memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.ZQuery) + HDR_LEN);
           CntZQry++;   
           send_msg(tx_buff); 
         break; 
   }
}

void utc_request( void )
{       
   switch(DataType){
      case NC_BINARY:  // navcore binary
         make_tx_header(206, 7);   
         *((short *)(tx_buff+10)) = 0;        /* Request Data For All SV's */
         cnt206++;
         send_msg(tx_buff);
         break; 
      case Z_BINARY:  // zodiac binary
         memset(&ZMsg.ZQuery, 0, sizeof(ZMsg.ZQuery));
         ZMsg.ZQuery.MsgID    = 1042;
         ZMsg.ZQuery.MsgFlags = Q_BIT | ANREQ;
         BldZQuery(&ZMsgBuf, &ZMsg.ZQuery);  
         memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.ZQuery) + HDR_LEN);
         CntZQry++;   
         send_msg(tx_buff); 
         break; 
   }
}


void BIT_tests( void )
{       
   switch(DataType){
      case NC_BINARY:  // navcore binary
         make_tx_header(202, 8);
         cnt202++;      
         send_msg(tx_buff);
         break; 
      case Z_BINARY:  // zodiac binary
         memset(&ZMsg.Msg1300, 0, sizeof(ZMsg.Msg1300));
         BldZMsg(&ZMsgBuf, ZMsg, 1300);  
         memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.Msg1300) + HDR_LEN);
         Cnt1300++; 
         send_msg(tx_buff);
         break; 
      case Z_NMEA:  // zodiac nmea
         sprintf(nmea_msg, "$PRWIIBIT,");
         cntibit++;  
         send_nmea_msg(nmea_msg); 
         break;
   }
}

void cold_start()
{
   // send cold start enable message  
   switch(DataType){
      case NC_BINARY:  // navcore binary
         make_tx_header(201, 36);
         *((short *)(tx_buff+10)) = (!ColdStartDisable) << 6;       /* Cold Start Bit */
         cnt201++;
         send_msg(tx_buff); 
         break;
      case NC_NMEA:  // nmea
         sprintf(nmea_msg, "$PRWIINIT,V,,A,,,,,,,,,,,");
         cntinit++;  
         send_nmea_msg(nmea_msg); 
         wait(1.5f);
         break;
      case Z_BINARY:  // zodiac binary
         memset(&ZMsg.Msg1216, 0, sizeof(ZMsg.Msg1216));
         ColdStartData.Disable = (tUSHORT) ColdStartDisable;
         ZMsg.Msg1216.TimeOut  = (tSHORT)  ColdStartTimeOut;
         BldZMsg(&ZMsgBuf, ZMsg, 1216);  
         memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.Msg1216) + HDR_LEN); 
         Cnt1216++;
         send_msg(tx_buff);
         break;
   }
}

void elev_mask(void)
{
   short valid;

   // satellite elevation mask angle
   clear_message_line();
   _settextposition(CMDR, CMDC);  
          
   switch(DataType){
      case NC_BINARY:  // navcore binary
         mask_angle = ask_float("ELEVATION MASK (-15 - +35 DEG) -> ", 10, &valid);
         clear_message_line(); 
         if(!valid) return;

         mask_num = (short) (mask_angle / 5);
         make_tx_header(208, 9);
         make_short(0x03, tx_buff, 6);   
         make_short(SVs01To16, tx_buff, 7);
         make_short(SVs17To32, tx_buff, 8);
         tx_buff[11] = (BYTE)(mask_num & 0x000f);   
         cnt208++;
         send_msg(tx_buff);
         break; 
      case Z_BINARY:  // zodiac binary
         mask_angle = ask_short("ELEVATION MASK (0 - 90 DEG) -> ", 10, &valid);
         clear_message_line(); 
         if(!valid) return;       
         // map to zodiac variables  
         memset(&ZMsg.Msg1212, 0, sizeof(ZMsg.Msg1212)); 
         ZMsg.Msg1212.ElevationMask = (tSHORT) (mask_angle * D2R * 1E+03);  
         BldZMsg(&ZMsgBuf, ZMsg, 1212);  
         memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.Msg1212) + 10);
         Cnt1212++;
         send_msg(tx_buff);
         break; 
   }
}

void init_2D(void)
{           
   // send 3 SV from acquisition message  
   switch(DataType){  
      case NC_BINARY:   // navcore binary
         make_tx_header(201, 36);        
         if(Auto_2D){
            // set 3SV Nav, lat, lon, alt valid bits
            *(short*)(tx_buff+10) = 0x24;      
         }
         else{ 
            // reset 3SV Nav bit and set lat, lon, alt valid bits 
            *(short*)(tx_buff+10) = 0x04;           
         } 
         *(short*)(tx_buff+10) = 0x0A;  
         make_F40(lat, tx_buff, 16);
         make_F40(lon, tx_buff, 19);
         make_F24(AltInput, tx_buff, 22);
         cnt201++;
         send_msg(tx_buff);
         break;
      case NC_NMEA:   // nmea
         if(Auto_2D){
            sprintf(nmea_msg, "$PRWIINIT,V,A,,,,,,,,,,,,");
         }
         else{
            sprintf(nmea_msg, "$PRWIINIT,V,V,,,,,,,,,,,,");    
         } 
         cntinit++; 
         send_nmea_msg(nmea_msg);  
         wait(1.5f);
         break;  
      default:
         break;
   }
}

void init_height(void)
{
   short valid;

   // get input parameters for altitude input
   clear_message_line();
   if((DataType == NC_BINARY) || (DataType == NC_NMEA) || (DataType == Z_NMEA)){
      AltInput = ask_dbl("NEW ALTITUDE (M) -> ", ref_alt, &valid);
      clear_message_line();   
      if(!valid) return;  
   }
   else if(DataType == Z_BINARY){
      ClearRAM = ask_short("CLEAR ALTITUDE IN RAM (0=NO, 1=YES) -> ", 0, &valid);
      clear_message_line();   
      if(!valid) return;
      ClearEEPROM = ask_short("CLEAR ALTITUDE IN EEPROM (0=NO, 1=YES) -> ", 0, &valid);
      clear_message_line();   
      if(!valid) return;
      
      // only input altitude if not clearing storage
      if(!ClearRAM & !ClearEEPROM){
         AltInput = ask_dbl("NEW ALTITUDE (M) -> ", ref_alt, &valid);
         clear_message_line();   
         if(!valid) return;
         AltStdDev = ask_dbl("ALTITUDE STANDARD DEVIATION (M) -> ", 0, &valid);
         clear_message_line();   
         if(!valid) return;    
         ForceUse = ask_short("FORCE ALTITUDE USE (0=NO, 1=YES) -> ", 0, &valid);
         clear_message_line();   
         if(!valid) return;
         MSLSelect = ask_short("ALTITUDE REFERENCE (0=GEODETIC, 1=MSL) -> ", 0, &valid);
         clear_message_line();   
         if(!valid) return;
         StoreRAM = ask_short("STORE ALTITUDE IN RAM (0=NO, 1=YES) -> ", 1, &valid);
         clear_message_line();   
         if(!valid) return;
         StoreEEPROM = ask_short("STORE ALTITUDE IN EEPROM (0=NO, 1=YES) -> ", 0, &valid);
         clear_message_line();
         if(!valid) return;   
      }
   }
  
   // build altitde input message
   switch(DataType){  
      case NC_BINARY:
         if(In_Nav_Mode){        
            // Amended Altitude
            make_tx_header(204, 8);
            make_F24(AltInput, tx_buff, 6);
            cnt204++;
            send_msg(tx_buff);
         }
         else{  
            // Altitude Hold 
            init_2D();           
         }
         break;    
         
      case NC_NMEA:
      case Z_NMEA:
         sprintf(nmea_msg, "$PRWIINIT,V,,,,,,,%lf,,,,,,",AltInput); 
         cntinit++; 
         send_nmea_msg(nmea_msg);  
         wait(1.5f);
         break;     
         
      case Z_BINARY:  
         memset(&ZMsg.Msg1219, 0, sizeof(ZMsg.Msg1219));
         AltInputData.ForceUse    = (tBOOL) ForceUse;
         AltInputData.MSLSelect   = (tBOOL) MSLSelect;
         AltInputData.StoreRAM    = (tBOOL) StoreRAM;
         AltInputData.StoreEEPROM = (tBOOL) StoreEEPROM;
         AltInputData.ClearRAM    = (tBOOL) ClearRAM;
         AltInputData.ClearEEPROM = (tBOOL) ClearEEPROM;
         ZMsg.Msg1219.Altitude  = (tLONG)  (AltInput  * 1E+02);
         ZMsg.Msg1219.AltStdDev = (tLONG)  (AltStdDev * 1E+02);
         BldZMsg(&ZMsgBuf, ZMsg, 1219);
         memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.Msg1219) + HDR_LEN);
         Cnt1219++; 
         send_msg(tx_buff);
         break;
   }
}

void init_pos_vel(void)
{
   double lat = 0, lon = 0, alt = 0; 
   double VelocityE = 0, VelocityN = 0, VelocityU = 0; 
   double GroundSpeed = 0, TrueCourse  = 0, ClimbRate   = 0;

   double latmin, lonmin;
   short  latdeg, londeg;
   char   latdir, londir;

   short  valid;

   // prompt for lat and lon
   clear_message_line();
   lat  = ask_dbl("LATITUDE (+N/-S DEG) -> ", ref_lat, &valid);
   clear_message_line();  
   if(!valid) return;
   lon = ask_dbl("LONGITUDE (+E/-W DEG) -> ", ref_lon, &valid);
   clear_message_line();      
   if(!valid) return;
   alt  = ask_dbl("ALTITUDE (M) -> ", ref_alt, &valid);
   clear_message_line();   
   if(!valid) return;  
   GroundSpeed  = ask_dbl("GROUND SPEED (M/S) -> ", 0, &valid);
   clear_message_line();  
   if(!valid) return;  
   TrueCourse = ask_dbl("TRUE COURSE (0-360 DEG) -> ", 0, &valid);
   clear_message_line();      
   if(!valid) return;  
   ClimbRate  = ask_dbl("CLIMB RATE (M/S) -> ", 0, &valid);
   clear_message_line();     
   if(!valid) return;        
   
   clear_message_line(); 
   _outtext("INITIALIZING RECEIVER POSITION AND VELOCITY");
   wait(1.f);   
   clear_message_line();
      
   // special processing for initialization
   VelocityE = GroundSpeed * sin(TrueCourse * D2R);    
   VelocityN = GroundSpeed * cos(TrueCourse * D2R);    
   VelocityU = ClimbRate;    
  
   // send pos/vel initialization message
   switch(DataType){
      case NC_BINARY:   // navcore binary
         make_tx_header(201, 36);    
         // set lat, lon, alt, speed and course valid bits
         *(short*)(tx_buff+10) = 0x0A; 
         make_F40(lat * D2R, tx_buff, 16);
         make_F40(lon * D2R, tx_buff, 19);
         make_F24(alt, tx_buff, 22); 
         make_F24(VelocityE, tx_buff, 24);
         make_F24(VelocityN, tx_buff, 26);
         make_F24(VelocityU, tx_buff, 28); 
         cnt201++;
         send_msg(tx_buff);
         break;
      case NC_NMEA:   // nmea message
      case Z_NMEA:
         if(lat < 0.){
            latdir = 'S'; 
            lat = -lat;
         }
         else{    
            latdir = 'N';
         }
         if(lon < 0.){
            londir = 'W'; 
            lon = -lon;
         }
         else{    
            londir = 'E';
         } 
       
         // convert latitude and longitude to deg/min and send
         latdeg = (short) lat;
         latmin = (lat - (double) latdeg) * 60.f;                     
         londeg = (short) lon;
         lonmin = (lon - (double) londeg) * 60.f; 
  
         sprintf(nmea_msg, 
                 "$PRWIINIT,V,,,%02d%06.3lf,%c,%03d%06.3lf,%c,%lf,%lf,M,%lf,T,,",
                 latdeg, latmin, latdir, londeg, lonmin,londir, alt,
                 GroundSpeed, TrueCourse );   
         cntinit++;            
         send_nmea_msg(nmea_msg); 
         wait(1.5f);
         break;
 
      case Z_BINARY:   // zodiac binary
          memset(&ZMsg.Msg1200, 0, sizeof(ZMsg.Msg1200)); 
              
          ZMsg.Msg1200.Latitude    = (long)  (lat * 1E+08 * D2R); 
          ZMsg.Msg1200.Longitude   = (long)  (lon * 1E+08 * D2R); 
          ZMsg.Msg1200.Altitude    = (long)  (alt * 1E+02);        
          ZMsg.Msg1200.GroundSpeed = (unsigned long)  (GroundSpeed * 1E+02); 
          ZMsg.Msg1200.Course      = (unsigned short) (TrueCourse  * 1E+03 * D2R); 
          ZMsg.Msg1200.ClimbRate   = (short)          (ClimbRate   * 1E+02);
           
          // set lat, lon, alt, speed and course valid bits 
          //ZMsg.Msg1200.Validity    = 0x00B8;
          InitValidData.ForceTime        = 0;    
          InitValidData.GPSTimeValid     = 0;    
          InitValidData.UTCTimeValid     = 0; 
          InitValidData.LatLonValid      = 1; 
          InitValidData.AltitudeValid    = 1; 
          InitValidData.SpeedCourseValid = 1;
          InitValidData.MagneticCourse   = 1;
          InitValidData.ClimbRateValid   = 1;

          BldZMsg(&ZMsgBuf, ZMsg, 1200);
          memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.Msg1200) + HDR_LEN); 
          Cnt1200++;  
          send_msg(tx_buff);
          break;
       default:
          break;
   }       
} 

void init_time( void )
{
   char dbuffer [9];
   char tbuffer [9];   
   short  numdays [] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
   short  valid;

   // get PC time as a default
   _strdate(dbuffer);
   year  = atoi(dbuffer+6); dbuffer[5] = '\0';
   day   = atoi(dbuffer+3); dbuffer[2] = '\0';
   month = atoi(dbuffer);   
   _strtime(tbuffer);   
   tbuffer[5] = '\0';
   minutes = atoi(tbuffer+3); tbuffer[2] = '\0';
   hours   = atoi(tbuffer); 

   // prompt for time and date
   clear_message_line();
   utcoffset = ask_short("UTC TIME OFFSET (+/- 0-12 HOURS) -> ", utcoffset, &valid);
   clear_message_line(); 
   if(!valid) return; 
   hours = hours + utcoffset; 
   if(hours >23){
      hours -= 24;
      day   += 1;  
      if(day > numdays[month-1]){
         month += 1;  
         if(month >= 12){
            year += 1;
         } 
      }
   }
   clear_message_line(); 
   if(!valid) return;
   minutes = ask_short("UTC MINUTES (0-59) -> ", minutes, &valid);
   clear_message_line();
   hours = ask_short("UTC HOUR (0-23) -> ", hours, &valid);    
   clear_message_line();
   if(!valid) return;
   day = ask_short("UTC DAY (1-31) -> ", day, &valid);
   clear_message_line();
   if(!valid) return;
   month = ask_short("UTC MONTH (1-12) -> ", month, &valid);  
   clear_message_line();
   if(!valid) return; 
   
   if(year >= 80 && year <= 99)
      year += 1900;
   else
   	  year += 2000;
   	  
   year = ask_short("UTC YEAR (1980-2079) -> ", year, &valid);
   clear_message_line(); 
   if(!valid) return;

   if (year < 0 || (year > 99 && year < 1980) || year > 2079) return;

   if (year < 1980) {
      if (year < 80) {
         year += 2000;
      }
      else {
         year += 1900;
      }
   }

   clear_message_line(); 
   _outtext("INITIALIZING RECEIVER TIME");
   wait(1.f);  

   // send time initialization message 
   switch(DataType){  
      case NC_BINARY:   // navcore binary
         make_tx_header(201, 36);
         *(short*)(tx_buff+10) = 0x04;         /* &valid Time */
         tx_buff[58] = (BYTE)hours;
         tx_buff[60] = (BYTE)minutes;
         tx_buff[62] = (BYTE)seconds;
         tx_buff[64] = (BYTE)day;
         tx_buff[66] = (BYTE)month;
         *((short *)(tx_buff+68)) = year;
         cnt201++; 
         send_msg(tx_buff);
         break;
      
      case NC_NMEA:   // nmea message
      case Z_NMEA:
         sprintf(nmea_msg, "$PRWIINIT,V,,,,,,,,,,,,%02d%02d%02d,%02d%02d%02d",
                           hours, minutes, seconds, day, month, (year%100));
         cntinit++;  
         send_nmea_msg(nmea_msg); 
         wait(1.5f);      
         break;
      
      case Z_BINARY:   // zodiac binary
          memset(&ZMsg.Msg1200, 0, sizeof(ZMsg.Msg1200));  

          ZMsg.Msg1200.UTCDay     = (unsigned short) day; 
          ZMsg.Msg1200.UTCMonth   = (unsigned short) month; 
          ZMsg.Msg1200.UTCYear    = (unsigned short) year;        
          ZMsg.Msg1200.UTCHours   = (unsigned short) hours; 
          ZMsg.Msg1200.UTCMinutes = (unsigned short) minutes; 
          ZMsg.Msg1200.UTCSeconds = (unsigned short) 0.;
           
          // set UTC time valid bits with force 
          InitValidData.ForceTime        = 1;    
          InitValidData.GPSTimeValid     = 0;    
          InitValidData.UTCTimeValid     = 1; 
          InitValidData.LatLonValid      = 0; 
          InitValidData.AltitudeValid    = 0; 
          InitValidData.SpeedCourseValid = 0;
          InitValidData.MagneticCourse   = 0;
          InitValidData.ClimbRateValid   = 0;
          
          BldZMsg(&ZMsgBuf, ZMsg, 1200);
          memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.Msg1200) + HDR_LEN); 
          Cnt1200++;
          send_msg(tx_buff);
          break;  
          
       default:
          break;
   }       
}

void init_timemark( void )
{
   short  timingmode = 0;  // 1=timing mode set, 0=normal mode
   short  selfsurvey = 0;  // 1=self survey mode set, 0=user input mode
   short  valid;
   unsigned long   CableDelay = 0;    // nanosec delay in cable
   long   SurveyPositionX     = 0;    // 1E+02 * meters
   long   SurveyPositionY     = 0;    // 1E+02 * meters
   long   SurveyPositionZ     = 0;    // 1E+02 * meters

   // prompt for initmode bits 0 and 1
   clear_message_line();
   timingmode = ask_short("TIMING MODE  (1=ON 0=OFF) -> ", timingmode, &valid);
   clear_message_line(); 
   if(!valid) return; 

   if ( timingmode ){  // only get this bit if timing mode selected
     CableDelay = (unsigned long)ask_long("CABLE DELAY (nanoseconds) -> ", CableDelay, &valid);
     clear_message_line();
     if(!valid) return;

     selfsurvey = ask_short("SELF SURVEY MODE (1=ON 0=OFF) -> ", selfsurvey, &valid);
     clear_message_line();
     if(!valid) return;
   }
   
   // make sure Initmode bits are logical
   if (selfsurvey) selfsurvey = 0x01;
   if (timingmode) timingmode = 0x01;
   
   if (timingmode && !selfsurvey){
     SurveyPositionX = ask_long("SURVEY X POSITION (meters) -> ", SurveyPositionX, &valid);
     clear_message_line();
     if(!valid) return;

     SurveyPositionY = ask_long("SURVEY Y POSITION (meters) -> ", SurveyPositionY, &valid);
     clear_message_line();
     if(!valid) return;

     SurveyPositionZ = ask_long("SURVEY Z POSITION (meters) -> ", SurveyPositionZ, &valid);
     clear_message_line();
     if(!valid) return;
   }

   clear_message_line(); 
   _outtext("INITIALIZING UTC TIMEMARK");
   wait(1.f);  

   // send timemark initialization message 
   switch(DataType){  

      case Z_BINARY:   // zodiac binary
          memset(&ZMsg.Msg1208, 0, sizeof(ZMsg.Msg1208));  

          ZMsg.Msg1208.CableDelay        = (tULONG) CableDelay; 
          ZMsg.Msg1208.SurveyPositionX   = (tLONG) SurveyPositionX; 
          ZMsg.Msg1208.SurveyPositionY   = (tLONG) SurveyPositionY; 
          ZMsg.Msg1208.SurveyPositionZ   = (tLONG) SurveyPositionZ;        
           
          // set UTC timemark mode bits with force 
          InitModeData.TimingMode       = timingmode;    
          InitModeData.SelfSurveyMode   = selfsurvey;    
          
          BldZMsg(&ZMsgBuf, ZMsg, 1208);
          memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.Msg1208) + HDR_LEN); 
          Cnt1208++;
          send_msg(tx_buff);
          break;  
          
       default:
          break;
   }       
}

void init_dr( void )
{
   // Global data for dr parameters
   extern tUSHORT DRSpeedSigma;      //standard deviation
   extern tUSHORT DRSpeedTimeTag;    //resolution  (100us)
   extern tUSHORT DRSpeedLatency;
   extern tUSHORT usDRFLAGWORD;      //flag
   extern tSHORT  sGyroScale;        //scale factor       (5.63063063e-3 deg/sec count)
   extern tSHORT  sGyroBias;         //bias               (221.424 deg/sec)
   extern tUSHORT usWheelTickRate;   //rate               (2485 ticks/km)

   short   valid, i;
   float   fDRSpeedSigma   = (float) ((DRSpeedSigma   * 1.0)/100.0);   // 1E+02 * meters/sec
   float   fDRSpeedTimeTag = (float) ((DRSpeedTimeTag * 1.0)/100.0);   // 1E+02 * ms
   float   fDRSpeedLatency = (float) ((DRSpeedLatency * 1.0)/100.0);   // 1E+02 * ms
   float   fReserved[5];
   char    cUseReservedWords;
   tUSHORT usLOCALDRFLAGWORD;      //local copy of flags


   // prompt for standard deviation
   clear_message_line();
   fDRSpeedSigma = ask_float("DR SPEED STANDARD DEVIATION (0..100.00) -> ", fDRSpeedSigma, &valid);
   DRSpeedSigma = (tUSHORT) ((fDRSpeedSigma*100.0)+0.5); 
   clear_message_line(); 
   if(!valid) return; 

   // prompt for time tag resolution
   clear_message_line();
   fDRSpeedTimeTag = ask_float("DR SPEED DATA TIME TAG RESOLUTION (0..655.35) -> ", fDRSpeedTimeTag, &valid);
   DRSpeedTimeTag = (tUSHORT) ((fDRSpeedTimeTag*100.0)+0.5);
   clear_message_line(); 
   if(!valid) return; 

   // prompt for initmode latency
   clear_message_line();
   fDRSpeedLatency = ask_float("DR SPEED LATENCY (0..655.35) -> ", fDRSpeedLatency, &valid);
   DRSpeedLatency = (tUSHORT) ((fDRSpeedLatency*100.0)+0.5);
   clear_message_line(); 
   if(!valid) return; 


   // clear all reserved words
   for(i=0; i<5; i++){
     fReserved[i]  = 0;
   }
   
   // get local copy of current flags
   usLOCALDRFLAGWORD = usDRFLAGWORD;
   
   // prompt to use reserve words and set reserved word Bit0 - flag
   clear_message_line();
   cUseReservedWords = (usLOCALDRFLAGWORD & 0x01) ? 'Y' : 'N';
   cUseReservedWords = ask_char("USE RESERVED WORDS BIT0 (y/n) -> ", cUseReservedWords, &valid);
   clear_message_line(); 
   if(!valid) return; 
   
   if((cUseReservedWords=='y') || (cUseReservedWords=='Y')){
     usDRFLAGWORD = 1; // set bit0  
     
     // prompt for reserve word 1 - dr heading rate scale factor
     clear_message_line();
     fReserved[1] = (float) (sGyroScale/100000.0);
     fReserved[1] = ask_float("DR HEADING RATE SCALE FACTOR (0..655.35) -> ", fReserved[1], &valid);
     sGyroScale = (tUSHORT) (fReserved[1]*100000.0);
     clear_message_line(); 
     if(!valid) return; 

     // prompt for reserve word 2 - dr heading rate bias
     clear_message_line();
     fReserved[2] = (float) (sGyroBias/100.0);
     fReserved[2] = ask_float("DR HEADING RATE BIAS (0..655.35) -> ", fReserved[2], &valid);
     sGyroBias = (tUSHORT) (fReserved[2]*100.0) ;
     clear_message_line(); 
     if(!valid) return;
   }
     
   // prompt to use reserve words and set reserved word Bit1 - flag
   clear_message_line();
   cUseReservedWords = (usLOCALDRFLAGWORD & 0x02) ? 'Y' : 'N';
   cUseReservedWords = ask_char("USE RESERVED WORDS BIT1 (y/n) -> ", cUseReservedWords, &valid);
   clear_message_line(); 
   if(!valid) return; 
   
   if((cUseReservedWords=='y') || (cUseReservedWords=='Y')){
     usDRFLAGWORD |= 2; // set bit1  
     
     
     // prompt for reserve word 3 - wheel tick rate
     clear_message_line();
     fReserved[3] = (float) usWheelTickRate;
     fReserved[3] = ask_float("WHEEL TICK RATE (0..655.35) -> ", fReserved[3], &valid);
     usWheelTickRate = (tUSHORT) (fReserved[3]) ;
     clear_message_line(); 
     if(!valid) return;
     
   }
   
   clear_message_line(); 
   _outtext("INITIALIZING DR");
   wait(1.f);  

   // send DR initialization message 
   switch(DataType){  

      case Z_BINARY:   // zodiac binary
          memset(&ZMsg.Msg1270, 0, sizeof(ZMsg.Msg1270));  

          // 0.5 has been added to some of the below float values before casting
          // to ensure that the values are rounded up.
          ZMsg.Msg1270.DRSpeedSigma     = DRSpeedSigma; 
          ZMsg.Msg1270.DRSpeedTimeTag   = DRSpeedTimeTag; 
          ZMsg.Msg1270.DRSpeedLatency   = DRSpeedLatency; 
          ZMsg.Msg1270.Reserved[0]      = usDRFLAGWORD; 
          ZMsg.Msg1270.Reserved[1]      = sGyroScale;
          ZMsg.Msg1270.Reserved[2]      = sGyroBias;
          ZMsg.Msg1270.Reserved[3]      = usWheelTickRate; 
          ZMsg.Msg1270.Reserved[4]      = (tUSHORT) fReserved[4]; 
           
          BldZMsg(&ZMsgBuf, ZMsg, 1270);
          memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.Msg1270) + HDR_LEN); 
          Cnt1270++;
          send_msg(tx_buff);
          break;  
          
       default:
          break;
   }       
}


void make_tx_header(short Message, short WordCount )
{
   short i, TotalBytes;

   TotalBytes = (WordCount) * 2;
  
   if (WordCount > 5) WordCount -= 6;
   else WordCount = 0;

   for(i=0;i<TotalBytes;i++) tx_buff[i] = 0;   /* Clear Buffer */
   *((unsigned short *)(tx_buff+0))= 0x81FF;     /* Header */
   *((short *)(tx_buff+2))= Message;             /* MSG ID */
   *((short *)(tx_buff+4))= WordCount;           /* Word Cnt */
   tx_buff[7] = 0x10;                          /* Request Ack */
}

void message_enable( short message, short turnon )
{
   make_tx_header(250, 8);

   *((short *)(tx_buff+10)) = message;           /* Message Number */

   if( turnon )
       tx_buff[12] = 0x01;                    /* Turn On Message */

}                   

int ResetReceiver(short Special)
{
   unsigned short InvalidateRAM;
   unsigned short InvalidateEEPROM;
   unsigned short InvalidateRTC;
   unsigned short ForceColdStart;
   unsigned short InvalidateRAM_vEEPROM = 0;
//   unsigned short InvalidateEEPROM_FREQSTD = 0;
   unsigned short EEPROMClearOnlyFS = 0;
   unsigned short EEPROMKeepFSCubic = 0;
   unsigned short EEPROMKeepFSTable = 0;
   unsigned short EEPROMKeepFSCal   = 0;
   unsigned short EEPROMKeepAlmanac = 0;
   unsigned short EEPROMKeepDRNav   = 0;

   char   ctemp;
   short  valid;

   // reset counters and send reset message
   reset_count();   
 
   if (DataType == NC_BINARY){  // navcore binary
      make_tx_header(201, 36);
      *((short*)(tx_buff+10)) = 0x10; // Reset bit
      cnt201++; 
      send_msg(tx_buff); 
   }
   else if (DataType == Z_BINARY){  // zodiac binary
      clear_message_line();
      ctemp = ask_char("INVALIDATE RAM (Y,N) -> ", 'N', &valid);
      clear_message_line();  
      if(!valid) return 0;     
      if(toupper(ctemp) == 'Y'){
         InvalidateRAM = 1;
      }
      else{
         InvalidateRAM = 0;
      }

      // CLEAR EEPROM.
      InvalidateEEPROM = 0;
      if (Special){
         ctemp = ask_char("INVALIDATE EEPROM (Y,N) -> ", 'N', &valid);
         clear_message_line();  
         if(!valid) return 0;     
         if(toupper(ctemp) == 'Y'){
            InvalidateEEPROM = 1;
         }
         else
         {
            InvalidateEEPROM = 0;
         }
         
         // Only Check for segments to keep if Clear EEPROM Requested.
         if (InvalidateEEPROM == 1)
         {               
            
            // EEPROM Clear Only Freq Std Segments.
            ctemp = ask_char("CLEAR ONLY EEPROM FREQ STD SEGMENTS (Y,N) -> ", 'N', &valid);
            clear_message_line();  
            if (!valid) return 0;     
            if (toupper(ctemp) == 'Y')
            {
               EEPROMClearOnlyFS = 1;
            }

/*
//PCM v5.7. Not active for now.  
            // Do NOT clear Only FS Segments
            // Check for other Keep areas
            else
            {
               // EEPROM Keep Freq Std Cubic.
               ctemp = ask_char("KEEP EEPROM FREQ STD CUBIC (Y,N) -> ", 'N', &valid);
               clear_message_line();  
               if (!valid) return 0;     
               if (toupper(ctemp) == 'Y')
               {
                  EEPROMKeepFSCubic = 1;
               }

               // EEPROM Keep Freq Std Table.
               ctemp = ask_char("KEEP EEPROM FREQ STD TABLE (Y,N) -> ", 'N', &valid);
               clear_message_line();  
               if (!valid) return 0;     
               if (toupper(ctemp) == 'Y')
               {
                  EEPROMKeepFSTable = 1;
               }

               // EEPROM Keep Freq Std Cal.
               ctemp = ask_char("KEEP EEPROM FREQ STD CAL (Y,N) -> ", 'N', &valid);
               clear_message_line();  
               if (!valid) return 0;     
               if (toupper(ctemp) == 'Y')
               {
                  EEPROMKeepFSCal = 1;
               }

               // EEPROM Keep Alamanc
               ctemp = ask_char("KEEP EEPROM ALMANAC (Y,N) -> ", 'N', &valid);
               clear_message_line();  
               if (!valid) return 0;     
               if (toupper(ctemp) == 'Y')
               {
                  EEPROMKeepAlmanac = 1;
               }

               // EEPROM Keep DR Navigation
               ctemp = ask_char("KEEP EEPROM DR NAV DATA (Y,N) -> ", 'N', &valid);
               clear_message_line();  
               if (!valid) return 0;     
               if (toupper(ctemp) == 'Y')
               {
                  EEPROMKeepDRNav = 1;
               }

            } // Check for other Keep areas
  */
         } // Clear EEPROM OK. Check for any segments to keep.
                  
      } // CLEAR EEPROM.

      ctemp = ask_char("INVALIDATE RTC (Y,N) -> ", 'N', &valid);
      clear_message_line();  
      if(!valid) return 0;     
      if(toupper(ctemp) == 'Y'){
         InvalidateRTC = 1;
      }
      else{
         InvalidateRTC = 0;
      }
      ctemp = ask_char("FORCE COLD START (Y,N) -> ", 'N', &valid);
      clear_message_line();  
      if(!valid) return 0;     
      if(toupper(ctemp) == 'Y'){
         ForceColdStart = 1;
      }
      else{
         ForceColdStart = 0;
      }

      // Clear RAM except for Virtual EEPROM data.
      if (Special)
      {
         ctemp = ask_char("INVALIDATE RAM EXCEPT vEEPROM (Y,N) -> ", 'N', &valid);
         clear_message_line();  
         if(!valid) return 0;     
         if(toupper(ctemp) == 'Y')
         {
            InvalidateRAM_vEEPROM = 1;
         }
      } // Clear RAM except for Virtual EEPROM data.


      clear_message_line();

      memset(&ZMsg.Msg1303, 0, sizeof(ZMsg.Msg1303));
            
      CommandWord.InvalidateRAM    = InvalidateRAM;
      CommandWord.InvalidateEEPROM = InvalidateEEPROM;
      CommandWord.InvalidateRTC    = InvalidateRTC;
      CommandWord.ForceColdStart   = ForceColdStart;
//      CommandWord.InvalidateEEPROM_FREQSTD = InvalidateEEPROM_FREQSTD;
      CommandWord.InvalidateRAM_vEEPROM = InvalidateRAM_vEEPROM;

      CommandWord.EEPROMClearOnlyFS= EEPROMClearOnlyFS;
      CommandWord.EEPROMKeepFSCubic= EEPROMKeepFSCubic;
      CommandWord.EEPROMKeepFSTable= EEPROMKeepFSTable;
      CommandWord.EEPROMKeepFSCal  = EEPROMKeepFSCal;  
      CommandWord.EEPROMKeepAlmanac= EEPROMKeepAlmanac;
      CommandWord.EEPROMKeepDRNav  = EEPROMKeepDRNav;  


      BldZMsg(&ZMsgBuf, ZMsg, 1303);
      memcpy(tx_buff, &ZMsgBuf, sizeof(ZMsg.Msg1303) + HDR_LEN);
      Cnt1303++;
      send_msg(tx_buff);
   }

   else if ((DataType == NC_NMEA) || (DataType == Z_NMEA)){  // NMEA-0183
      sprintf(nmea_msg, "$PRWIINIT,A,,,,,,,,,,,,,");  
      cntinit++; 
      send_nmea_msg(nmea_msg); 
      wait(1.5f); 
   }
   return 1;
} 

void static_nav(void)
{
   make_tx_header(201, 36);

   clear_message_line();
   _settextposition(cmdline,1);

   if(StaticNavEnable){
      *((short *)(tx_buff+10)) = 0x80;    /* Static Bit */
      _outtext("TURNING STATIC NAV ON ");
   }
   else{
     *((short *)(tx_buff+10)) = 0x00;     /* Static Bit */
      _outtext("TURNING STATIC NAV OFF ");
   }
} 

